<?php

namespace App\Services;

use App\Utilities\AppClass;
use DataTables;
use App\Services\FeedbackService;
use Illuminate\Support\Facades\Validator;

class DataTableService
{

    const STATUS_ACTIVE = 1;
    const STATUS_INACTIVE = 0;

    const CREATED_AT_SCREENS = [
        'feedback',
        'dashboard-detailed-report',
        'complaint',
    ];

    const DOWNLOAD_RECORD_SCREENS = [
        'feedback',
        // 'complaint',
    ];

    const RESOLVE_AUTH_SCREENS = ['complaint'];

    const REPORT_ACTION_VIEWS = ['enquiry-report', 'just-dial-leads'];

    public $feedbackService;

    public function __construct(FeedbackService $feedbackService)
    {
        $this->args = new AppClass();
        $this->feedbackService = $feedbackService;
    }

    public function getDataTable($data, $args = null)
    {
        $this->args = $args;
        $appType = getAppTypes();
        $this->typeActive = $appType->typeActive;
        $this->typeInactive = $appType->typeInactive;
        $this->typeDelete = $appType->typeDelete;
        $this->typeStatusUpdate = $appType->typeStatusUpdate;

        $ret = Datatables::of($data)->addIndexColumn()
            ->addColumn('status', function ($row) {
                if ($row->status == self::STATUS_ACTIVE) {
                    $btn = '<span class="label text-success d-flex"><span class="dot-label bg-success mr-1"></span>Active</span>';
                } else {
                    $btn = '<span class="label text-muted d-flex"><span class="dot-label bg-gray-300 mr-1"></span>Inactive</span>';
                }

                return $btn;
            });

        if (isset($this->args->wrapTxt)) {
            $ret->addColumn('wrapTxt', function ($row) {
                $btn = '<div class="wrapDiv">' . $row->wrapText . '</div>';
                return $btn;
            });
        }

        if (isset($this->args->showFiles)) {

            $ret->addColumn('downloadFile', function ($row) {
                $btn = $this->generateDownloadLink($row->downloadUrl);
                return $btn;
            });
        }

        $ret->addColumn('action', function ($row) {

            //for caller page only
            if (!$this->args->dropDownDesign) {

                $btn = '<button class="btn btn-sm btn-primary click-to-call mr-1" data-unique-id="' . encrypt($row->{$this->args->tableColumn}) . '" data-url="' . route('caller.makeCall') . '" data-redirect-url="">
                            <i class="fe fe-phone"></i>
                        </button>';
                // $btn = '<button class="btn btn-sm btn-primary click-to-call mr-1" data-unique-id="' . encrypt($row->{$this->args->tableColumn}) . '" data-url="' . route('caller.makeCall') . '" data-redirect-url="' . route('feedback.createNew', encrypt($row->{$this->args->tableColumn})) . '">
                //             <i class="fe fe-phone"></i>
                //         </button>';
                // $btn = '<button class="btn btn-sm btn-primary mr-1">
                //                 <i class="fe fe-phone"></i>
                //             </button>';

                $btn .= '<a href="' . route($this->args->createRouteName, encrypt($row->{$this->args->tableColumn})) . '"
                    class="btn btn-sm btn-info mr-1">
                    <i class="fe fe-edit-2"></i>
                    </a>';

                if ($this->args->delete) {
                    $btn .= '<button class="btn btn-sm btn-danger skip_reason" data-unique-id="' . encrypt($row->{$this->args->tableColumn}) . '" data-url="' . route('UpdateStatusAjax') . '">
                                    <i class="fe fe-trash"></i>
                                    </button>';
                }
            } else {
                $btn = '<div class="dropdown">
                <button aria-expanded="false" aria-haspopup="true" class="btn btn-sm ripple btn-primary dropdown-toggle" data-toggle="dropdown" id="dropdownMenuButton" type="button">Action <i class="fas fa-caret-down ml-1"></i></button>
                <div  class="dropdown-menu tx-13">';

                // if ($this->args->create) {
                //     $btn .= '<a class="dropdown-item" href="' . route($this->args->createRouteName, encrypt($row->{$this->args->tableColumn})) . '">Create</a>';
                // }

                //only for feedback master
                if ($this->args->showCriticalPoints) {
                    $btn .= '<a class="dropdown-item" href="' . route('feedback-master.mapCriticalPoints', encrypt($row->{$this->args->tableColumn})) . '">Map Critical Points</a>';
                }

                if ($this->args->view) {

                    $btn .= '<a class="dropdown-item" href="' . route($this->args->showRouteName, encrypt($row->{$this->args->tableColumn})) . '">View</a>';
                }

                if ($this->args->edit) {
                    $btn .= '<a class="dropdown-item" href="' . route($this->args->editRouteName, encrypt($row->{$this->args->tableColumn})) . '">Edit</a>';
                }

                if ($this->args->delete) {
                    $btn .= '<a class="dropdown-item deleteRow" href="' . route($this->args->deleteRouteName, ['id' => encrypt($row->{$this->args->tableColumn}), 'type' => $this->typeDelete, 'name' => $this->args->screenName]) . '">Delete</a>';
                }

                if ($this->args->export) {
                    if ($this->args->screenName == 'call-mapping') {
                        $btn .= '<a class="dropdown-item" href="' . route('exportExcelById', ['id' => encrypt($row->{$this->args->tableColumn}), 'name' => $this->args->screenName]) . '">Export Excel</a>';
                    }
                }

                if ($this->args->active && $row->status == self::STATUS_INACTIVE) {
                    $btn .= '<a class="dropdown-item" href="' . route($this->args->activeRouteName, ['id' => encrypt($row->{$this->args->tableColumn}), 'type' => $this->typeActive, 'name' => $this->args->screenName]) . '">Active</a>';
                } else if ($this->args->inactive && $row->status == self::STATUS_ACTIVE) {
                    $btn .= '<a class="dropdown-item" href="' . route($this->args->inactiveRouteName, ['id' => encrypt($row->{$this->args->tableColumn}), 'type' => $this->typeInactive, 'name' => $this->args->screenName]) . '">Inactive</a>';
                }

                if (in_array($this->args->screenName, self::RESOLVE_AUTH_SCREENS) && $this->isResolveAuth($row->{$this->args->tableColumn})) {

                    $btn .= '<a class="dropdown-item resolve_complaint" data-unique-id="' . encrypt($row->{$this->args->tableColumn}) . '" data-url="' . route('resolveFeedback') . '" data-redirect-url="' . route('complaint.index') . '">Resolve</a>';
                }

                $btn .= '</div>
                        </div>';
            }

            return $btn;
        });

        if (isset($this->args->screenName) && in_array($this->args->screenName, self::CREATED_AT_SCREENS)) {
            $ret->addColumn('dateTime', function ($row) {
                $html = AppDateTimeFormat($row->created_at);
                if (isset($row->call_user)) {
                    $html .= '<br>' . $row->call_user;
                }
                return $html;
            });
        }

        if (isset($this->args->screenName) && in_array($this->args->screenName, self::DOWNLOAD_RECORD_SCREENS)) {

            $ret->addColumn('downloadRecordings', function ($row) {

                $btn = '';
                if (isset($this->args->showRecording) && $this->args->showRecording && $row->recordings) {
                    $btn .= '<a href="' . route("downloadAttachments", ["type" => "call-record", "file" => $row->id]) . '"><i class="fe fe-download attachment_icon"></i></a>';
                    // $recordList = explode(',', $row->recordings);
                    // if (count($recordList) > 0) {
                    //     foreach ($recordList as $recording) {
                    //         if ($recording != '') {
                    //             $btn .= '<a href="' . route("downloadAttachments", ["type" => "call-record", "file" => $recording]) . '"><i class="fe fe-download attachment_icon"></i></a>';
                    //         }
                    //     }
                    // }
                }
                return $btn;
            });
        }

        $ret->rawColumns(['action', 'status', 'downloadRecordings', 'wrapTxt', 'dateTime', 'downloadFile']);
        return $ret->make(true);
    }


    function generateDownloadLink($url)
    {
        $validator = Validator::make(['url' => $url], [
            'url' => 'required|url',
        ]);

        if (!$validator->fails()) {
            return '<a href="' . $url . '" target="_blank"><i class="fa fa-download attachment_icon"></i></a>';
        }

        return '';
    }

    public function getDataTableReports($data, $args = null)
    {
        $this->args = $args;

        $ret = Datatables::of($data)->addIndexColumn();

        if ($this->args->showTimeStamp == true) {
            $ret->addColumn('dateTime', function ($row) {
                $html = AppDateTimeFormat($row->created_at);
                if (isset($row->created_user)) {
                    $html .= '<br>' . $row->created_user;
                }
                return $html;
            });
        }

        if (isset($this->args->showAction)) {
            $ret->addColumn('action', function ($row) {

                $btn = '<div class="dropdown">
                <button aria-expanded="false" aria-haspopup="true" class="btn btn-sm ripple btn-primary dropdown-toggle" data-toggle="dropdown" id="dropdownMenuButton" type="button">Action <i class="fas fa-caret-down ml-1"></i></button>
                <div  class="dropdown-menu tx-13">';

                if ($this->args->view) {

                    if (in_array($this->args->screenName, self::REPORT_ACTION_VIEWS) && $row->status == 1) {
                        $btn = '';
                        $btn .= '<button class="btn btn-sm btn-danger leadStatusModal" data-unique-id="' . encrypt($row->{$this->args->tableColumn}) . '" data-url="' . route('UpdateStatusAjax') . '">
                                    <i class="fe fe-edit"></i>
                                    </button>';
                    } else if ($this->args->screenName == 'ongoing-calls') {
                        $btn .= '<a class="dropdown-item" href="' . route('updateCallStatus', ['id' => encrypt($row->{$this->args->tableColumn})]) . '">Update</a>';
                    } else {
                        return '';
                    }
                }

                $btn .= '</div>
                        </div>';

                return $btn;
            });
        }

        if (isset($this->args->showrecordings)) {

            $ret->addColumn('downloadRecordings', function ($row) {

                $validator = Validator::make(['url' => $row->recordings], [
                    'url' => 'required|url',
                ]);

                $btn = '';
                if (!$validator->fails()) {
                    $btn = '<a href="' . $row->recordings . '"><i class="fe fe-download attachment_icon"></i></a>';
                }
                return $btn;
            });
        }
        $ret->rawColumns(['action', 'status', 'downloadRecordings', 'dateTime']);
        return $ret->make(true);
    }

    public function isResolveAuth($id)
    {
        $data = $this->feedbackService->getFeedbackDetails($id);
        $data->isResolvedAuthority = $this->feedbackService->isResolvedAuthority($data);
        return $data->isResolvedAuthority;
    }
}
