<?php

namespace App\Services\Reports;

use App\Models\Masters\BranchMaster;
use App\Models\Masters\CallSkipReason;
use App\Models\Sales;
use App\Services\DashboardService;
use App\Services\ExcelExportService;
use App\Services\Reports\ReportService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use DB;
use Illuminate\Support\Collection;
use stdClass;

class BranchSummaryService
{
    private $reportService;
    private $excelExportService;
    private $dashboardService;

    public function __construct(ReportService $reportService, ExcelExportService $excelExportService, DashboardService $dashboardService)
    {
        $this->reportService = $reportService;
        $this->excelExportService = $excelExportService;
        $this->dashboardService = $dashboardService;
    }

    function querySql1($args)
    {
        $newArgs = clone $args;
        $sql = $this->dashboardService->coreAgentAssignFollowDetails($newArgs);
        $sql->select(
            'beneficiary_import.branch_id',
            DB::raw('COUNT(beneficiary_import.id) as allocatedCalls'),
        )
            ->groupBy('beneficiary_import.branch_id');
        $data = $sql->get();
        return $data;
    }

    function querySql3($args)
    {
        $newArgs = clone $args;
        $sql = $this->dashboardService->coreAgentCallDetails($newArgs);
        $sql->select(
            'beneficiary_import.branch_id',

            // Count total and categorized calls (distinct feedback.id to prevent duplicate counts)
            DB::raw('COUNT(DISTINCT feedback.id) as dialledCalls'),
            DB::raw('COUNT(DISTINCT CASE WHEN feedback.call_status = 1 THEN feedback.id END) as connectedCalls'),
            DB::raw('COUNT(DISTINCT CASE WHEN feedback.call_status = 2 THEN feedback.id END) as notConnectedCalls'),

            // Summing Call Durations Correctly (Distinct call_history.id prevents duplicate durations)
            DB::raw('SEC_TO_TIME(SUM(DISTINCT CASE WHEN feedback.call_status = 1 THEN TIME_TO_SEC(COALESCE(call_history.call_duration, "00:00:00")) END)) as connectedCallsDuration'),
            DB::raw('SEC_TO_TIME(AVG(DISTINCT CASE WHEN feedback.call_status = 1 THEN TIME_TO_SEC(COALESCE(call_history.call_duration, "00:00:00")) END)) as avgConnectedCallsDuration'),
            DB::raw('SEC_TO_TIME(SUM(DISTINCT CASE WHEN feedback.call_status = 2 THEN TIME_TO_SEC(COALESCE(call_history.call_duration, "00:00:00")) END)) as notConnectedCallsDuration'),

            // Sum of connected calls on converted sales (fixes duplication issue)
            DB::raw('SEC_TO_TIME(SUM(DISTINCT CASE WHEN feedback.call_status = 1 AND sales.id IS NOT NULL THEN TIME_TO_SEC(COALESCE(call_history.call_duration, "00:00:00")) END)) as connectedOnConverted'),
            DB::raw('SEC_TO_TIME(AVG(DISTINCT CASE WHEN feedback.call_status = 1 AND sales.id IS NOT NULL THEN TIME_TO_SEC(COALESCE(call_history.call_duration, "00:00:00")) END)) as avgConnectedOnConverted'),
            DB::raw('SEC_TO_TIME(
                (SELECT AVG(call_duration_sec) 
                 FROM (SELECT DISTINCT feedback.id, TIME_TO_SEC(COALESCE(call_history.call_duration, "00:00:00")) as call_duration_sec
                       FROM call_history
                       LEFT JOIN beneficiary_import ON beneficiary_import.id = call_history.customer_id
                       LEFT JOIN feedback ON feedback.beneficiary_ref_id = beneficiary_import.id
                       LEFT JOIN sales ON sales.beneficiary_id = beneficiary_import.id
                       WHERE feedback.call_status = 1 AND sales.id IS NOT NULL) as unique_durations
                )
            ) as avgConnectedOnConverted')

        )
            ->leftJoin('call_history', 'feedback.id', '=', 'call_history.feedback_id')
            ->leftJoin(DB::raw('(SELECT DISTINCT beneficiary_id, id FROM sales) as sales'), 'beneficiary_import.id', '=', 'sales.beneficiary_id') // Ensures only one sales record per beneficiary
            ->groupBy('beneficiary_import.branch_id');

        $data = $sql->get();
        return $data;
    }

    function querySql5($args)
    {
        $newArgs = clone $args;
        $sql = $this->dashboardService->coreAgentCallDetails($newArgs);
        $sql->select(
            'beneficiary_import.branch_id',
            'call_skip_reason.id',
            'call_skip_reason.name',
            DB::raw('COUNT(feedback.id) as count')
        )
            ->leftJoin('call_skip_reason', 'call_skip_reason.id', 'feedback.call_skip_reason')
            ->groupBy('beneficiary_import.branch_id', 'call_skip_reason.id');
        $data = $sql->get();
        return $data;
    }

    function querySql4($args)
    {
        $newArgs = clone $args;
        $sql = Sales::select(
            'branch_id',
            DB::raw('COUNT(id) as totalSalesQty'),
            DB::raw('SUM(amount) as totalSalesValue'),
            DB::raw('COUNT(IF(beneficiary_id IS NOT NULL, id, NULL)) as salesConvertedQty'),
            DB::raw('SUM(IF(beneficiary_id IS NOT NULL, amount, 0)) as salesConvertedValue'),
            DB::raw('COUNT(IF(beneficiary_id IS NULL, id, NULL)) as salesNotConvertedQty'),
            DB::raw('SUM(IF(beneficiary_id IS NULL, amount, 0)) as salesNotConvertedValue'),
        );
        $sql->where('invoice_date', '>=', $newArgs->fromDate)
            ->where('invoice_date', '<=', $newArgs->toDate);
        if ($newArgs->branch != 'All') {
            $sql->where('branch_id', $newArgs->branch);
        }
        $data = $sql->first();
        return $data;
    }

    public function getData(Request $request)
    {
        $args = $this->reportService->processrequestFilters($request);
        $fromDate = Carbon::parse($args->years . '-' . $args->months . '-01')->format('Y-m-d');
        $toDate = Carbon::parse($fromDate)->endOfMonth()->format('Y-m-d');
        $args->fromDate = $fromDate;
        $args->toDate = $toDate;
        $args->agent = 'All';
        if ($args->branch == '') {
            $args->branch = 'All';
        }

        //getting all call skip reasons
        $callReason = CallSkipReason::whereStatus(CallSkipReason::STATUS_ACTIVE)->get();

        //getting branches
        $branchSql = BranchMaster::select('id', 'name')->whereStatus(BranchMaster::STATUS_ACTIVE);
        if ((int) $args->branch > 0) {
            $branchSql->where('id', $args->branch);
        }
        $branchList = $branchSql->get();
        $queryResult1 = $this->querySql1($args);
        $queryResult3 = $this->querySql3($args);
        $queryResult4 = $this->querySql4($args);
        $queryResult5 = $this->querySql5($args);

        $result = new stdClass;
        $result->records = new Collection();
        $result->title = 'Telecalling Report' . AppDateFormat($fromDate) . ' - ' . AppDateFormat($toDate);
        $result->connected = $callReason->where('is_answered', 1);
        $result->notConnected = $callReason->where('is_answered', '!=', 1);

        foreach ($branchList as $branch) {

            $item = new stdClass;

            $queryBranchData1 = $queryResult1->where('branch_id', $branch->id)->first();
            $queryBranchData3 = $queryResult3->where('branch_id', $branch->id)->first();
            $queryBranchData4 = $queryResult4->where('branch_id', $branch->id)->first();

            $allocatedCalls = $queryBranchData1->allocatedCalls ?? 0;
            $dialledCalls = $queryBranchData3->dialledCalls ?? 0;
            $connectedCalls = $queryBranchData3->connectedCalls ?? 0;
            $notConnectedCalls = $queryBranchData3->notConnectedCalls ?? 0;
            $totalSalesQty = $queryBranchData4->totalSalesQty ?? 0;
            $totalSalesValue = $queryBranchData4->totalSalesValue ?? 0;
            $salesConvertedQty = $queryBranchData4->salesConvertedQty ?? 0;
            $salesConvertedValue = $queryBranchData4->salesConvertedValue ?? 0;
            $salesNotConvertedQty = $queryBranchData4->salesNotConvertedQty ?? 0;
            $salesNotConvertedValue = $queryBranchData4->salesNotConvertedValue ?? 0;

            $dialledPercentage = 0;
            if ($allocatedCalls > 0 && $dialledCalls > 0) {
                $dialledPercentage = round($dialledCalls / $allocatedCalls * 100);
            }

            $connectedPercentage = 0;
            if ($connectedCalls > 0 && $dialledCalls > 0) {
                $connectedPercentage = round($connectedCalls / $dialledCalls * 100);
            }


            $item->connected = new Collection();
            $item->notConnected = new Collection();

            foreach ($callReason as $row) {
                $callData = $queryResult5->where('branch_id', $branch->id)->where('name', $row->name)->first();
                $node = new stdClass;
                $node->name = $row->name;
                $node->value = $callData->count ?? 0;
                $node->properConn = null;
                if (strtolower($row->dispositions) == 'interested' || strtolower($row->dispositions) == 'customer callback') {
                    $node->properConn = 1;
                }
                if ($row->is_answered == 1) {
                    $item->connected->push($node);
                } else {
                    $item->notConnected->push($node);
                }
            }

            $properCallsPercentage = 0;
            $properConnSum = $item->connected->where('properConn', 1)->sum('value');
            if ($properConnSum > 0 && $connectedCalls > 0) {
                $properCallsPercentage = round($properConnSum / $connectedCalls * 100);
            }

            $item->branchName = $branch->name;
            $item->allocatedCalls = $allocatedCalls;
            $item->dialledCalls = $dialledCalls;
            $item->dialledPercentage = $dialledPercentage;
            $item->connectedCalls = $connectedCalls;
            $item->notConnectedCalls = $notConnectedCalls;
            $item->connectedPercentage = $connectedPercentage;
            $item->properCallsPercentage = $properCallsPercentage;
            $item->interestedCount = $properConnSum;
            $item->interestedPercentage = $properCallsPercentage;
            $item->totalSalesQty = $totalSalesQty;
            $item->totalSalesValue = $totalSalesValue;
            $item->salesConvertedQty = $salesConvertedQty;
            $item->salesConvertedValue = $salesConvertedValue;
            $item->salesNotConvertedQty = $salesNotConvertedQty;
            $item->salesNotConvertedValue = $salesNotConvertedValue;
            $item->connectedCallsDuration = $queryBranchData3->connectedCallsDuration ?? 0;
            $item->avgConnectedCallsDuration = $queryBranchData3->avgConnectedCallsDuration ?? 0;
            $item->notConnectedCallsDuration = $queryBranchData3->notConnectedCallsDuration ?? 0;
            $item->connectedOnConverted = $queryBranchData3->connectedOnConverted ?? 0;
            $item->avgConnectedOnConverted = $queryBranchData3->avgConnectedOnConverted ?? 0;
            $result->records->add($item);
        }

        return $result;
    }

    function getReportExcel(Request $request)
    {
        $data = $this->getData($request);
        $args = $this->reportService->processrequestFilters($request);
        $params = clone $args;
        $params->type = 'sales-analysis-report-excel';
        $params->branchLabel = $this->reportService->getBranchById($args->branch);
        return $this->excelExportService->exportExcel($params, $data);
    }
}
