<?php

namespace App\Services\Reports;

use App\Models\Sales;
use App\Services\DashboardService;
use App\Services\ExcelExportService;
use App\Services\Reports\ReportService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use DB;
use Illuminate\Support\Collection;
use stdClass;

class SalesAnalysisReportService
{
    private $reportService;
    private $excelExportService;
    private $dashboardService;

    public function __construct(ReportService $reportService, ExcelExportService $excelExportService, DashboardService $dashboardService)
    {
        $this->reportService = $reportService;
        $this->excelExportService = $excelExportService;
        $this->dashboardService = $dashboardService;
    }

    function getQuery1($args)
    {
        // DB::enableQueryLog();
        $sql = $this->dashboardService->coreAgentAssignFollowDetails($args);
        $data = $sql->select(
            DB::raw('COUNT(beneficiary_import.id) as allocatedCalls'),
        )->first();
        // $queries = DB::getQueryLog();
        // dd($queries);
        return $data;
    }

    function getQuery2($args)
    {
        $sql = $this->dashboardService->coreAgentCallDetails($args);
        $data = $sql->select(
            DB::raw('COUNT(CASE WHEN feedback.call_status = 1 THEN feedback.id END) as connectedCalls'),
        )->first();
        return $data;
    }

    function getQuery3($args)
    {
        $sql = Sales::select(
            DB::raw('COUNT(IF(beneficiary_id IS NOT NULL, id, NULL)) as salesConverted'),
            DB::raw('AVG(IF(beneficiary_id IS NOT NULL, amount, NULL)) as avgSalesConverted'),
            DB::raw('SUM(IF(beneficiary_id IS NOT NULL, amount, 0)) as totalSalesConverted'),
            DB::raw('COUNT(IF(beneficiary_id IS NULL, id, NULL)) as salesNotConverted'),
            DB::raw('AVG(IF(beneficiary_id IS NULL, amount, NULL)) as avgSalesNotConverted'),
            DB::raw('SUM(IF(beneficiary_id IS NULL, amount, 0)) as totalSalesNotConverted'),
            DB::raw('COUNT(id) as salesTotal'),
            DB::raw('AVG(amount) as avgSalesTotal'),
            DB::raw('SUM(amount) as totalSalesTotal')
        );
        $sql->where('invoice_date', '>=', $args->fromDate)
            ->where('invoice_date', '<=', $args->toDate);
        if ($args->branch != 'All') {
            $sql->where('branch_id', $args->branch);
        }
        $data = $sql->first();
        return $data;
    }

    public function getData(Request $request)
    {
        $args = $this->reportService->processrequestFilters($request);
        $dates = $this->reportService->getLastTwoMonths($request);
        $branch = 'All';
        $agent = 'All';
        if ($args->branch) {
            $branch = $args->branch;
        }
        if ($args->agent) {
            $agent = $args->agent;
        }

        $result = new stdClass;
        $result->tableOne = new Collection();
        $result->tableHeader = new Collection();
        $result->tableTwo = new Collection();

        foreach ($dates->data as $key => $row) {
            $newArgs = new stdClass;
            $newArgs->fromDate = $row['start'];
            $newArgs->toDate = $row['end'];
            $newArgs->branch = $branch;
            $newArgs->agent = $agent;
            $queryResult1 = $this->getQuery1($newArgs);
            $queryResult2 = $this->getQuery2($newArgs);
            $queryResult3 = $this->getQuery3($newArgs);

            $result->tableHeader->add($row['fullText']);

            $salesConvOnConnCalls = 0;
            $connCallRatio = 0;

            if ($queryResult3->salesConverted && $queryResult1->allocatedCalls) {
                $salesConvOnConnCalls = $queryResult3->salesConverted / $queryResult1->allocatedCalls * 100;
            }
            if ($queryResult2->connectedCalls && $queryResult1->allocatedCalls) {
                $connCallRatio = $queryResult2->connectedCalls / $queryResult1->allocatedCalls * 100;
            }

            $tableOneClass = new stdClass;
            $tableOneClass->salesConvOnConnCalls = round($salesConvOnConnCalls, 1);
            $tableOneClass->allocatedCalls = $queryResult1->allocatedCalls;
            $tableOneClass->connCallRatio = round($connCallRatio, 1);
            $tableOneClass->connTotalAll = $queryResult2->connectedCalls;
            $tableOneClass->salesConversion = $queryResult3->salesConverted;
            $tableOneClass->avgSalesPrice = round($queryResult3->avgSalesConverted, 2);
            $tableOneClass->salesConvertedAllocatedCalls = $queryResult3->totalSalesConverted;
            $result->tableOne->add($tableOneClass);

            $tableTwoClass = new stdClass;
            $tableTwoClass->totalSales = new stdClass;
            $tableTwoClass->connCalls = new stdClass;
            $tableTwoClass->notConnCalls = new stdClass;

            $tableTwoClass->totalSales->qty = $queryResult3->salesTotal;
            $tableTwoClass->totalSales->value = $queryResult3->totalSalesTotal;

            $tableTwoClass->connCalls->qty = $queryResult3->salesConverted;
            $tableTwoClass->connCalls->value = $queryResult3->totalSalesConverted;

            $tableTwoClass->notConnCalls->qty = $queryResult3->salesNotConverted;
            $tableTwoClass->notConnCalls->value = $queryResult3->totalSalesNotConverted;

            $result->tableTwo->add($tableTwoClass);
        }

        $tableTwoClass = new stdClass;
        $tableTwoClass->totalSales = new stdClass;
        $tableTwoClass->connCalls = new stdClass;
        $tableTwoClass->notConnCalls = new stdClass;

        $tableTwoClass->totalSales->qty = $result->tableTwo->sum(function ($item) {
            return $item->totalSales->qty ?? 0;
        });
        $tableTwoClass->totalSales->value = $result->tableTwo->sum(function ($item) {
            return $item->totalSales->value ?? 0;
        });

        $tableTwoClass->connCalls->qty = $result->tableTwo->sum(function ($item) {
            return $item->connCalls->qty ?? 0;
        });
        $tableTwoClass->connCalls->value = $result->tableTwo->sum(function ($item) {
            return $item->connCalls->value ?? 0;
        });

        $tableTwoClass->notConnCalls->qty = $result->tableTwo->sum(function ($item) {
            return $item->notConnCalls->qty ?? 0;
        });
        $tableTwoClass->notConnCalls->value = $result->tableTwo->sum(function ($item) {
            return $item->notConnCalls->value ?? 0;
        });

        $result->tableTwo->prepend($tableTwoClass);
        // dd($result);
        return $result;
    }

    function getReportExcel(Request $request)
    {
        $data = $this->getData($request);
        $args = $this->reportService->processrequestFilters($request);
        $params = clone $args;
        $params->type = 'sales-analysis-report-excel';
        $params->branchLabel = $this->reportService->getBranchById($args->branch);
        return $this->excelExportService->exportExcel($params, $data);
    }
}
