<?php

namespace App\Services\Transactions;

use App\Models\ApplicationSettings;
use App\Models\CRMTickets;
use App\Models\EnquiryHistory;
use App\Models\ExcelCampaign;
use App\Models\JustDialLeads;
use App\Models\LeadsUpdateHistory;
use App\Models\Masters\AgentMaster;
use App\Models\Masters\BranchLeadsAssign;
use App\Models\Masters\BranchMaster;
use App\Models\Masters\CallSkipReason;
use App\Models\Masters\ClassMaster;
use App\Models\Transactions\BeneficiaryImport;
use App\Services\CommonService;
use App\Services\ExcelCampaignService;
use App\Services\ExcelExportService;
use App\Utilities\AppClass;
use Carbon\Carbon;
use DB;
use Log;
use Illuminate\Http\Request as HttpRequest;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use stdClass;

class CallCenterService
{

    private $commonService;
    private $excelExportService;
    private $excelCampaignService;

    const HTTP_STATUS_CODE_SUCCESS = 200;
    const ENQUIRY_IGNORE_DESCRIPTIONS = ['Model not available with us', 'Not eligible for finance'];

    public function __construct(CommonService $commonService, ExcelExportService $excelExportService, ExcelCampaignService $excelCampaignService)
    {
        $this->commonService = $commonService;
        $this->excelExportService = $excelExportService;
        $this->excelCampaignService = $excelCampaignService;
    }

    public function coreCallerSql()
    {
        $sql = DB::table('beneficiary_import')
            ->leftJoin('branch_master', 'branch_master.id', 'beneficiary_import.branch_id')
            ->select(
                'beneficiary_import.status',
                'beneficiary_import.description',
                'beneficiary_import.id',
                'beneficiary_import.customer_name',
                'beneficiary_import.mobile_number',
                'branch_master.name as branchName',
                DB::raw('DATE_FORMAT(beneficiary_import.callback_on, "%d/%m/%Y %h:%i %p") as callbackOn'),
                DB::raw('DATE_FORMAT(beneficiary_import.invoice_date, "%d-%m-%Y") as date'),
                DB::raw('CASE WHEN beneficiary_import.source = 1 THEN "Tele Calling" WHEN beneficiary_import.source = 2 THEN "Just Dial" WHEN beneficiary_import.source = 3 THEN "CRM Ticket" WHEN beneficiary_import.source = 5 THEN "SAP" ELSE "N/A" END as leadSource'),
            )
            ->whereNull('beneficiary_import.deleted_at');
        return $sql;
    }

    public function getData(HttpRequest $request, $type = null)
    {
        // $districtId = [];
        $authDetails = Auth::user();
        $role = $authDetails->role;
        // $userType = $authDetails->user_type;
        // $branchId = [];
        // if (!is_null($role) && $userType == User::USER_TYPE_AGENT && $role != RoleMaster::ROLE_ADMIN) {
        //     $agentId = $authDetails->reference_id;
        //     $branchId = AgentMapping::where('agent_id', $agentId)->pluck('branch_id');
        // }

        // if (!is_null($role) && count($districtId) == 0 && $role != RoleMaster::ROLE_ADMIN) {
        //     $data = [];
        //     return $data;
        // }

        $sql = $this->coreCallerSql();
        $sql->leftJoin('agent_master', 'agent_master.id', 'beneficiary_import.agent_id');
        $sql->addSelect('agent_master.name as agentName');
        if ($type == 'caller') {
            $sql->where('beneficiary_import.status', BeneficiaryImport::STATUS_ACTIVE);
        }

        if ($type == 'caller' && isset($request->branch) && $request->branch > 0) {
            $sql->where('beneficiary_import.branch_id', $request->branch);
        }
        return $sql;
    }

    function getDialerCounts(HttpRequest $request, $type = '')
    {
        $item = new stdClass();

        $allocatedTelecalling = 0;
        $callbackTelecalling = 0;
        $allocatedJustDial = 0;
        $callbackJustDial = 0;
        $allocatedCRM = 0;
        $callbackCRM = 0;
        $allocatedSAP = 0;
        $callbackSAP = 0;
        $allocatedCampaign = 0;
        $callbackCampaign = 0;
        $allocatedHappyCalls = 0;
        $callbackHappyCalls = 0;

        //telecalling lead
        $request = $request->merge(['teleCalling' => 1]);
        $teleSql = $this->getDialerList($request, 'counts');
        if ($teleSql) {
            if ($type == 'app-dialer') {
                $request = $request->merge(['isCallback' => 1]);
                $teleSqlCallback = $this->getDialerList($request, 'counts');
                $callbackTelecalling = $teleSqlCallback->count();
            }
            $allocatedTelecalling = $teleSql->count();
        }
        $request = $request->merge(['isCallback' => null]);
        $request = $request->merge(['teleCalling' => null]);

        //just dial leads
        $request = $request->merge(['isJustDial' => 1]);
        $justSql = $this->getDialerList($request, 'counts');
        if ($justSql) {
            if ($type == 'app-dialer') {
                $request = $request->merge(['isCallback' => 1]);
                $justSqlCallback = $this->getDialerList($request, 'counts');
                $callbackJustDial = $justSqlCallback->count();
            }
            $allocatedJustDial = $justSql->count();
        }
        $request = $request->merge(['isCallback' => null]);
        $request = $request->merge(['isJustDial' => null]);

        //crm leads
        $request = $request->merge(['isCRMLead' => 1]);
        $crmSql = $this->getDialerList($request, 'counts');
        if ($crmSql) {
            if ($type == 'app-dialer') {
                $request = $request->merge(['isCallback' => 1]);
                $crmSqlCallback = $this->getDialerList($request, 'counts');
                $callbackCRM = $crmSqlCallback->count();
            }
            $allocatedCRM = $crmSql->count();
        }

        $request = $request->merge(['isCallback' => null]);
        $request = $request->merge(['isJustDial' => null]);
        $request = $request->merge(['isCRMLead' => null]);

        //sap leads
        $request = $request->merge(['isSAPLead' => 1]);
        $sapSql = $this->getDialerList($request, 'counts');
        if ($sapSql) {
            if ($type == 'app-dialer') {
                $request = $request->merge(['isCallback' => 1]);
                $sapSqlCallback = $this->getDialerList($request, 'counts');
                $callbackSAP = $sapSqlCallback->count();
            }
            $allocatedSAP = $sapSql->count();
        }

        //excel campaign
        $request = $request->merge(['isCallback' => null]);
        $request = $request->merge(['isJustDial' => null]);
        $request = $request->merge(['isCRMLead' => null]);
        $request = $request->merge(['isSAPLead' => null]);

        $request = $request->merge(['isExcelCampaign' => 1]);
        $camSql = $this->getDialerList($request, 'counts');
        if ($camSql) {
            if ($type == 'app-dialer') {
                $request = $request->merge(['isCallback' => 1]);
                $camSqlCallback = $this->getDialerList($request, 'counts');
                $callbackCampaign = $camSqlCallback->count();
            }
            $allocatedCampaign = $camSql->count();
        }

        //happy calls
        $request = $request->merge(['isCallback' => null]);
        $request = $request->merge(['isJustDial' => null]);
        $request = $request->merge(['isCRMLead' => null]);
        $request = $request->merge(['isSAPLead' => null]);
        $request = $request->merge(['isExcelCampaign' => null]);

        $request = $request->merge(['isHappyCalls' => 1]);
        $hapSql = $this->getDialerList($request, 'counts');
        if ($hapSql) {
            if ($type == 'app-dialer') {
                $request = $request->merge(['isCallback' => 1]);
                $hapSqlCallback = $this->getDialerList($request, 'counts');
                $callbackHappyCalls = $hapSqlCallback->get()->count();
            }
            $allocatedHappyCalls = $hapSql->get()->count();
        }

        //telecalling
        $item->totalTeleCalling = $allocatedTelecalling + $callbackTelecalling;
        $item->callbackTelecalling = $callbackTelecalling;
        $item->allocatedTelecalling = $allocatedTelecalling;
        //just dial
        $item->totalJustdial = $allocatedJustDial + $callbackJustDial;
        $item->allocatedJustDial = $allocatedJustDial;
        $item->callbackJustDial = $callbackJustDial;
        //crm
        $item->totalCRM = $allocatedCRM + $callbackCRM;
        $item->allocatedCRM = $allocatedCRM;
        $item->callbackCRM = $callbackCRM;
        //sap
        $item->totalSAP = $allocatedSAP + $callbackSAP;
        $item->allocatedSAP = $allocatedSAP;
        $item->callbackSAP = $callbackSAP;
        //excel campaign 
        $item->totalCampaign = $allocatedCampaign + $callbackCampaign;
        $item->allocatedCampaign = $allocatedCampaign;
        $item->callbackCampaign = $callbackCampaign;
        //happy calls
        $item->totalHappyCalls = $allocatedHappyCalls + $callbackHappyCalls;
        $item->allocatedHappyCalls = $allocatedHappyCalls;
        $item->callbackHappyCalls = $callbackCampaign;

        return response()->json($item);
    }

    public function getDialerList(HttpRequest $request, $type = null)
    {
        $item = new stdClass();
        $item->recordsTotal = 0;
        $item->recordsFiltered = 0;
        $item->data = new Collection();

        $user = $request->user();
        $agentId = $user->reference_id;
        $sql = $this->coreCallerSql();
        $condition = '=';
        $needAgent = true;

        //for happy calls
        if (isset($request->isHappyCalls) && $request->isHappyCalls) {
            $sql->where('beneficiary_import.source', BeneficiaryImport::SOURCE_TYPE_HAPPY_CALLS);
        }

        //
        if (isset($request->teleCalling) && $request->teleCalling) {
            $sql->where('beneficiary_import.source', BeneficiaryImport::SOURCE_TYPE_EXCEL);
        }

        //for fetching excel campaign leads
        // if (isset($request->isExcelCampaign) && $request->isExcelCampaign) {
        //     // $agent = $this->excelCampaignService->getExcelCampaignUserDetails($user);
        //     $agent = AgentMaster::select('branch_id')->find($agentId);
        //     if ($agent) {
        //         $needAgent = false;
        //         $sql->where('beneficiary_import.branch_id', $agent->branch_id);
        //         $sql->where('beneficiary_import.agent_id', $agentId);
        //         $sql->where('beneficiary_import.source', BeneficiaryImport::SOURCE_TYPE_EXCEL_CAMPAIGN);
        //     } else {
        //         if ($type == 'counts') {
        //             return null;
        //         }
        //         return response()->json($item);
        //     }
        // }

        if (isset($request->isExcelCampaign) && $request->isExcelCampaign) {
            $needAgent = false;
            $leadAccess = AgentMaster::select('vendor_campaign_lead_assign', 'branch_id')->find($agentId);
            if ($leadAccess && $leadAccess->vendor_campaign_lead_assign == 1) {
                $sql->where('beneficiary_import.branch_id', $leadAccess->branch_id);
                $sql->where('beneficiary_import.source', BeneficiaryImport::SOURCE_TYPE_EXCEL_CAMPAIGN);
            } else {
                if ($type == 'counts') {
                    return null;
                }
                return response()->json($item);
            }
        }

        //for fetching callback leads
        if (isset($request->isCallback) && $request->isCallback) {
            $condition = '!=';
            $startOfDay = Carbon::now()->startOfDay();
            $endOfDay = Carbon::now()->endOfDay();
            $sql->whereBetween('callback_on', [$startOfDay, $endOfDay]);
            // $sql->whereDate('callback_on', Carbon::now());
            $sql->where('beneficiary_import.skip_reason', CallSkipReason::REASON_CALL_BACK);
        }
        //for fetching just dial leads
        if (isset($request->isJustDial) && $request->isJustDial) {
            $needAgent = false;
            $leadAccess = BranchLeadsAssign::select('branch_id')
                ->where('user_id', $user->id)
                ->first();

            if ($leadAccess && $leadAccess->branch_id) {
                $sql->where('beneficiary_import.branch_id', $leadAccess->branch_id);
                $sql->where('beneficiary_import.source', BeneficiaryImport::SOURCE_TYPE_JUST_DIAL);
            } else {
                if ($type == 'counts') {
                    return null;
                }
                return response()->json($item);
            }
        }

        //for fetching crm leads
        if (isset($request->isCRMLead) && $request->isCRMLead) {
            $needAgent = false;
            $leadAccess = AgentMaster::select('crm_lead_assign', 'branch_id')->find($agentId);
            if ($leadAccess && $leadAccess->crm_lead_assign == 1) {
                $sql->where('beneficiary_import.branch_id', $leadAccess->branch_id);
                $sql->where('beneficiary_import.source', BeneficiaryImport::SOURCE_TYPE_CRM);
                $sql->join('crm_tickets', 'crm_tickets.id', 'beneficiary_import.crm_lead_id');
                // $sql->whereIn('crm_tickets.ticket_type', ['Sale Enquiry']);
                $sql->whereIn('crm_tickets.ticket_type', [1]);
            } else {
                if ($type == 'counts') {
                    return null;
                }
                return response()->json($item);
            }
        }

        //for fetching sap leads
        if (isset($request->isSAPLead) && $request->isSAPLead) {
            $needAgent = false;
            $leadAccess = AgentMaster::select('sap_lead_assign', 'branch_id')->find($agentId);
            if ($leadAccess && $leadAccess->sap_lead_assign == 1) {
                $sql->where('beneficiary_import.branch_id', $leadAccess->branch_id);
                $sql->where('beneficiary_import.source', BeneficiaryImport::SOURCE_TYPE_SAP);
                $sql->join('enquiry_history', 'enquiry_history.id', 'beneficiary_import.sap_lead_id');
                $sql->whereNotIn('enquiry_history.enquiry_reason', self::ENQUIRY_IGNORE_DESCRIPTIONS);
                $sql->addSelect('enquiry_history.amount');
            } else {
                if ($type == 'counts') {
                    return null;
                }
                return response()->json($item);
            }
        }

        if ($needAgent) {
            $sql->where('beneficiary_import.agent_id', $agentId);
        }

        $sql->where('beneficiary_import.status', $condition, BeneficiaryImport::STATUS_ACTIVE);

        $sql->addSelect('beneficiary_import.call_status');
        if ($type == 'counts') {
            return $sql;
        }
        $clone = clone $sql;
        if ($request->page && $request->limit) {
            $limit = $request->page * $request->limit;
            $skip = $limit - $request->limit;
            $sql->skip($skip)->take($limit);
        }
        $sql->orderBy('beneficiary_import.id');
        $data = $sql->get();
        if (isset($request->isHappyCalls) && $request->isHappyCalls) {
            $item->recordsTotal = $clone->get()->count();
        } else {
            $item->recordsTotal = $clone->count();
        }
        $item->recordsFiltered = $data->count();
        $item->data = $data;

        return response()->json($item);
    }

    public function getCallSkipReasons(HttpRequest $request)
    {
        $reason = CallSkipReason::select('id', 'name', 'applicable_for')
            ->whereIn('is_answered', [CallSkipReason::BOTH, $request->type])
            ->whereStatus(CallSkipReason::STATUS_ACTIVE)
            ->get();
        return $reason;
    }

    public function isNew($id = null) {}

    public function saveData(HttpRequest $request, $id = null) {}

    public function getDBColumnsImport()
    {
        return [
            'branch_id' => 'Branch',
            'invoice_date' => 'Date',
            'customer_name' => 'Customer Name',
            'mobile_number' => 'Mobile Number',
            'description' => 'Description',
            'class' => 'Class',
        ];
    }

    public function saveImportData(HttpRequest $request)
    {

        $response = new stdClass;
        $response->status = false;
        $response->msg = new Collection();

        try {

            DB::beginTransaction();

            $batchID = date('YmdHis');

            $rows = json_decode($request->get('rows'), true);
            $header = $request->get('headerColumn');

            $this->branchList = BranchMaster::whereStatus(BranchMaster::STATUS_ACTIVE)->select('id', 'code')->get();
            $this->classList = ClassMaster::whereStatus(ClassMaster::STATUS_ACTIVE)->select('id', 'name')->get();

            $importItems = new Collection();

            foreach ($rows as $rowKey => $row) {
                if ($rowKey > 0) {
                    $data = new stdClass;
                    $addItem = true;
                    foreach ($header as $headerKey => $headerRow) {
                        if (isset($row[$headerKey]) && $row[$headerKey] != '') {
                            $rowValue = $this->getMastersReference($headerRow, $row[$headerKey]);
                            if ($rowValue && $rowValue->value) {
                                $data->{$headerRow} = $rowValue->value;
                            } else {
                                $errorMsg = $row[$headerKey] . ' => ' . $rowValue->dataType . ' => not exists in database';
                                $response->msg->add($errorMsg);
                                $addItem = false;
                            }
                        }
                    }

                    if ($addItem) {
                        $importItems->add($data);
                    }
                }
            }

            if ($response->msg->count() == 0) {

                foreach ($importItems as $row) {

                    $date = str_replace('/', '-', $row->invoice_date);
                    $data = new BeneficiaryImport();
                    $data->batch_id = $batchID;
                    $data->customer_name = isIssetValue($row, 'customer_name');
                    $data->mobile_number = $this->commonService->formatMobileNo($row->mobile_number);
                    $data->invoice_date = Carbon::parse($date)->format('Y-m-d');
                    $data->description = isIssetValue($row, 'description');
                    $data->branch_id = isset($row->branch_id) ? $row->branch_id : null;
                    $data->class = isset($row->class) ? $row->class : null;
                    $data->import_type = BeneficiaryImport::IMPORT_TYPE_MANUAL;
                    $data->created_by = Auth::user()->id;
                    $data->source = BeneficiaryImport::SOURCE_TYPE_EXCEL;
                    $data->save();
                }

                //to assign call to users
                // Artisan::call('call:allocate');

                $successMsg = 'Data imported successfully';
                $response->msg->add($successMsg);
                $response->status = true;
            }
            DB::commit();
            return $response;
        } catch (\Exception $e) {
            DB::rollback();
            Log::info('customer Import Error : ' . $e);
            dd($e->getMessage());
            return $response;
        }
    }

    public function isBranchExists($rowValues)
    {
        foreach ($this->branchList as $branch) {
            if (formatString($rowValues) == formatString($branch->code)) {
                return $branch->id;
            }
        }
    }

    public function isClassExists($rowValues)
    {
        foreach ($this->classList as $class) {
            if (formatString($rowValues) == formatString($class->name)) {
                return $class->id;
            }
        }
    }

    public function getMastersReference($dbColumns, $rowValues)
    {
        $value = new AppClass();
        $value->dataType = null;
        $value->value = null;

        switch ($dbColumns) {
            case 'branch_id':
                $value->value = $this->isBranchExists($rowValues);
                $value->dataType = 'Branch';
                break;
            case 'class':
                $value->value = $this->isClassExists($rowValues);
                $value->dataType = 'Class';
                break;
            default:
                $value->value = $rowValues;
                break;
        }

        return $value;
    }

    public function getAgentPhoneNumber()
    {
        $authDetails = Auth::user();
        if ($authDetails) {
            $agentId = $authDetails->reference_id;
            if ($agentId) {
                return AgentMaster::find($agentId);
            }
        }
    }

    public function makeCall(HttpRequest $request)
    {
        $benId = $request->has('benId') ? decrypt($request->get('benId')) : '';

        if ($benId) {

            $agentDetails = $this->getAgentPhoneNumber();

            if ($agentDetails) {

                $beneficiaryDetails = BeneficiaryImport::find($benId);

                if ($beneficiaryDetails) {

                    $callParams = [
                        'phone_number' => addCountryCodeToPhone($beneficiaryDetails->mobile_number),
                        'agent_number' => addCountryCodeToPhone($agentDetails->phone),
                        'caller_id' => addCountryCodeToPhone($agentDetails->phone),
                    ];

                    $response = $this->makeApiCall($callParams);
                    Log::info('make call response : ' . json_encode($response));
                    // $response = [
                    //     'status' => true,
                    //     'msg' => 'Call Conncted Successfully',
                    // ];

                    if ($response && $response['status'] == true) {
                        // $beneficiaryDetails->status = BeneficiaryImport::STATUS_CALL_INITIATED;
                        // $beneficiaryDetails->save();
                    }
                } else {
                    $response = [
                        'status' => false,
                        'msg' => 'Beneficiary not found',
                    ];
                }
            } else {
                $response = [
                    'status' => false,
                    'msg' => 'Agent details not found',
                ];
            }
        }

        return $response;
    }

    public function initCall(HttpRequest $request)
    {
        Log::info('make call : ' . json_encode($request->all()));

        //for testing knowlarity calls
        // $agentNo = formatMobileNo($request->agentNo);
        // $customerNumber = '+918130411226';
        // if ($agentNo == '8588854926') {
        //     $customerNumber = '+919037099937';
        // }
        // $response = new AppClass();
        // $response->customerNumber =  $customerNumber;
        // Log::info('customer call initiated : ' . json_encode($response));
        // return $response;

        $response = new AppClass();
        if ($request->agentNo) {
            $agentNo = formatMobileNo($request->agentNo);
            $agentDetails = AgentMaster::where('phone', $agentNo)->first();
            if ($agentDetails) {
                $customerDetails = BeneficiaryImport::where('agent_id', $agentDetails->id)->where('call_status', 1)->orderBy('updated_at', 'desc')->first();
                if ($customerDetails) {
                    $response->customerNumber =  addCountryCodeToPhone($customerDetails->mobile_number);
                    Log::info('customer call initiated : ' . json_encode($response));
                } else {
                    Log::info('customer details not found in database');
                }
            } else {
                Log::info('agent number not found in database');
            }
        } else {
            Log::info('agent number not found in request');
        }

        return $response;
    }

    public function makeApiCall($callParams)
    {
        $response = [
            'status' => false,
            'msg' => 'Call not connected, Contact admin',
        ];

        $apiDetails = constants('CALL_API_DETAILS');

        if ($apiDetails) {

            $endpoint = $apiDetails['endPoint'];

            $callParams['is_promotional'] = false;
            $callParams['sr_number'] = $apiDetails['srNumber'];
            $callParams['api_key'] = $apiDetails['apiKey'];

            $client = new \GuzzleHttp\Client();
            $response = $client->request('GET', $endpoint, ['query' => $callParams]);
            $statusCode = $response->getStatusCode();

            if ($statusCode == self::HTTP_STATUS_CODE_SUCCESS) {
                $response = [
                    'status' => true,
                    'msg' => 'Call Conncted Successfully',
                ];
            }
        }

        return $response;
    }

    public function getCallbackList(HttpRequest $request)
    {
        $authDetails = Auth::user();
        $userId = $authDetails->id;

        $dateRange = $request->has('date_range') ? $request->get('date_range') : null;
        $dateRange = explode('-', $dateRange);
        $fromDate = Carbon::parse(trim($dateRange[0]))->format('Y-m-d');
        $toDate = Carbon::parse(trim($dateRange[1]))->format('Y-m-d');

        $sql = $this->getData($request);

        $sql->where(function ($query) use ($userId) {
            $query->orWhere(function ($query1) use ($userId) {
                $query1->where('beneficiary_import.skip_reason', BeneficiaryImport::SKIP_REASON_CALLBACK)
                    // ->where('beneficiary_import.updated_by', $userId)
                    ->where('beneficiary_import.status', BeneficiaryImport::STATUS_CALL_SKIPPED);
            });
        });

        if ($fromDate && $toDate) {
            $sql->whereDate('beneficiary_import.callback_on', '>=', $fromDate);
            $sql->whereDate('beneficiary_import.callback_on', '<=', $toDate);
        }

        $sql->addSelect(
            DB::raw('DATE_FORMAT(beneficiary_import.callback_on,"%d/%m/%Y %h:%i:%s %p") as callback_on'),
            DB::raw('DATE_FORMAT(invoice_date, "%d-%m-%Y") as date')
        );

        return $sql;
    }

    public function getCallAnotherList(HttpRequest $request)
    {
        $authDetails = Auth::user();
        $userId = $authDetails->id;
        $sql = $this->getData($request);
        $sql->where('beneficiary_import.status', BeneficiaryImport::STATUS_CALL_SKIPPED);
        $sql->where('beneficiary_import.skip_reason', BeneficiaryImport::SKIP_REASON_CALL_ANOTHER_NUMBER);
        $sql->where('beneficiary_import.updated_by', $userId);
        $sql->addSelect(
            DB::raw('beneficiary_import.another_number'),
            DB::raw('DATE_FORMAT(CURDATE(), "%d-%m-%Y") as date')
        );
        return $sql;
    }

    public function updateCallStatus(HttpRequest $request)
    {
        $files = [];

        $id = $request->has('callId') ? $request->get('callId') : null;

        if ($id) {

            $id = decrypt($id);

            $fileUploadTypes = [BeneficiaryImport::SKIP_REASON_MASTERDATA_CORRECTION, BeneficiaryImport::SKIP_REASON_CALL_DISCONNECTED, BeneficiaryImport::SKIP_REASON_CALL_ANOTHER_NUMBER];

            $reason = $request->has('call_reason') ? $request->get('call_reason') : null;
            $callbackOn = $request->has('callback_date_time') ? $request->get('callback_date_time') : null;
            $anotherNumber = $request->has('call_another_number') ? $request->get('call_another_number') : null;

            if (in_array($reason, $fileUploadTypes)) {
                if ($request->file('recordings')) {

                    foreach ($request->file('recordings') as $key => $file) {
                        $fileName = time() . rand(1, 99) . '.' . $file->extension();
                        $file->move(public_path('uploads/recordings'), $fileName);
                        array_push($files, $fileName);
                    }
                }
            }

            if ($reason == BeneficiaryImport::SKIP_REASON_CALLBACK) {
                $callbackOn = Carbon::parse($callbackOn);
            }

            $data = BeneficiaryImport::find($id);
            $data->status = BeneficiaryImport::STATUS_CALL_SKIPPED;
            $data->skip_reason = $reason;
            $data->updated_by = Auth::user()->id;
            $data->callback_on = $callbackOn;
            $data->another_number = $anotherNumber;
            $data->recordings = implode(',', $files);
            $data->save();
            return true;
        }
    }

    public function updateCallSkipReasons(HttpRequest $request)
    {
        $files = [];
        $id = $request->has('callId') ? $request->get('callId') : null;
        $reason = $request->has('call_reason') ? $request->get('call_reason') : null;
        $callbackOn = $request->has('callback_date_time') ? $request->get('callback_date_time') : null;
        $anotherNumber = $request->has('call_another_number') ? $request->get('call_another_number') : null;
        if ($reason == BeneficiaryImport::SKIP_REASON_CALLBACK) {
            $callbackOn = Carbon::parse($callbackOn);
        }
        $data = BeneficiaryImport::find($id);
        $data->status = BeneficiaryImport::STATUS_CALL_SKIPPED;
        $data->skip_reason = $reason;
        $data->updated_by = Auth::user()->id;
        $data->callback_on = $callbackOn;
        $data->another_number = $anotherNumber;
        $data->recordings = implode(',', $files);
        if ($data->save()) {
            return $this->commonService->makeSuccessResponse('Call status updated successfully', $data);
        }
        return $this->commonService->makeErrorResponse('failed to update status');
    }

    public function makeCallApp(HttpRequest $request)
    {
        try {
            $benId = $request->callId;
            if ($benId) {
                $user = $request->user();
                if (!$user) {
                    return $this->commonService->makeErrorResponse('User details not found');
                }

                $agentDetails = AgentMaster::find($user->reference_id);
                if ($agentDetails) {

                    $beneficiaryDetails = BeneficiaryImport::find($benId);

                    //check any ongoing calls exists
                    $sql = BeneficiaryImport::where('call_status', 1)->where('id', '!=', $benId);
                    if ($beneficiaryDetails->source == BeneficiaryImport::SOURCE_TYPE_EXCEL) {
                        $sql->where('agent_id', $agentDetails->id);
                    } else {
                        $sql->where('source', $beneficiaryDetails->source);
                        $sql->where('branch_id', $beneficiaryDetails->branch_id);
                    }
                    $callCount = $sql->count();

                    if ($callCount > 0) {
                        return $this->commonService->makeErrorResponse('Previous call not closed yet.');
                    }


                    $beneficiaryDetails->call_status = 1;
                    $beneficiaryDetails->agent_id = $agentDetails->id;
                    $beneficiaryDetails->updated_at = Carbon::now();
                    $beneficiaryDetails->save();

                    //get call option
                    // 1 - for normal call
                    //2 - for knowlarity call
                    $settings = ApplicationSettings::where('setting_name', 'call_options')->first();

                    if ($settings->setting_value == '2') {

                        $return = new AppClass();
                        // $return->srNumber = constants('srNumber');
                        $return->customerNumber = constants('srNumber');
                        return $this->commonService->makeSuccessResponse('Call initiated successfully', $return);
                    } else {
                        $return = new AppClass();
                        $return->customerNumber = $beneficiaryDetails->mobile_number;
                        return $this->commonService->makeSuccessResponse('Call initiated successfully', $return);
                    }
                } else {
                    return $this->commonService->makeErrorResponse('Agent details not found');
                }
            }
            return $this->commonService->makeErrorResponse('customer id not found');
        } catch (\Exception $e) {
            Log::info('make call app error : ' . $e->getMessage());
            return $this->commonService->makeErrorResponse($e->getMessage());
        }
    }

    function saveRecordings(HttpRequest $request)
    {
        $path = 'uploads/recordings/';
        $fileArr = explode('.', $request->docName);
        $extension = $fileArr[1];
        if ($extension) {
            $extension = '.' . $extension;
            $fileName = $this->commonService->saveBase64StringToFile($request->document, $path, $extension);
            if ($fileName) {
                $data = new stdClass;
                $data->fileName = $fileName;
                return $this->commonService->makeSuccessResponse('Call status updated successfully', $data);
            }
        }
    }

    function convertTimeFormat($seconds)
    {
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $remainingSeconds = $seconds % 60;

        if ($hours > 0) {
            return "{$hours}:{$minutes}:{$remainingSeconds}";
        } elseif ($minutes > 0) {
            return "00:{$minutes}:{$remainingSeconds}";
        } else {
            return "00:00:{$remainingSeconds}";
        }
    }

    function saveCallLogs(HttpRequest $request)
    {
        // $agentId = $request->user()->reference_id;
        // if ($request->data) {
        //     foreach ($request->data as $row) {
        //         $dateTime = Carbon::parse($row['dateTime'])->format('Y-m-d H:i:s');
        //         $isLog = UsersCallLogs::where('agent_id', $agentId)
        //             ->where('customer_mobile', $row['phoneNumber'])
        //             ->where('call_date_time', $dateTime)
        //             ->count();
        //         if ($isLog == 0) {

        //             //check beneficiary exist in table
        //             $phoneNumber = $this->commonService->formatMobileNo($row['phoneNumber']);
        //             $beneficiaryDetails = BeneficiaryImport::where('mobile_number', $phoneNumber)->first();

        //             $time = $this->convertTimeFormat($row['duration']);
        //             $log = new UsersCallLogs();
        //             $log->agent_id = $agentId;
        //             $log->beneficiary_id = isset($beneficiaryDetails) ? $beneficiaryDetails->id : null;
        //             $log->customer_name = isset($row['name']) ? $row['name'] : null;
        //             $log->customer_mobile = isset($phoneNumber) ? $phoneNumber : null;
        //             $log->call_type = isset($row['type']) ? $row['type'] : null;
        //             $log->call_duration = $time;
        //             $log->call_date_time = $dateTime;
        //             $log->created_by = $agentId;
        //             $log->save();
        //         }
        //     }
        // }
        $data = new stdClass;
        return $this->commonService->makeSuccessResponse('Call logs updated successfully', $data);
    }

    function updateLeadStatus(HttpRequest $request)
    {
        try {
            DB::beginTransaction();
            $data = new stdClass;
            $justDialId = null;
            $crmId = null;
            $sapId = null;
            $excelId = null;

            $beneficiaryDetails = BeneficiaryImport::find($request->callId);
            if ($beneficiaryDetails) {
                if ($request->leadType == BeneficiaryImport::SOURCE_TYPE_JUST_DIAL) {
                    $justDial = JustDialLeads::find($beneficiaryDetails->jd_lead_id);
                    if ($justDial) {
                        $justDial->read_status = $request->status;
                        $justDial->save();
                        $justDialId = $justDial->id;
                    }
                } else if ($request->leadType == BeneficiaryImport::SOURCE_TYPE_CRM) {
                    $crm = CRMTickets::find($beneficiaryDetails->crm_lead_id);
                    if ($crm) {
                        $crm->read_status = $request->status;
                        $crm->save();
                        $crmId = $crm->id;
                    }
                } else if ($request->leadType == BeneficiaryImport::SOURCE_TYPE_SAP) {
                    $sap = EnquiryHistory::find($beneficiaryDetails->sap_lead_id);
                    if ($sap) {
                        $sap->read_status = $request->status;
                        $sap->save();
                        $sapId = $sap->id;
                    }
                } else if ($request->leadType == BeneficiaryImport::SOURCE_TYPE_EXCEL_CAMPAIGN) {
                    $excel = ExcelCampaign::find($beneficiaryDetails->excel_campaign_id);
                    if ($excel) {
                        $excel->read_status = $request->status;
                        $excel->save();
                        $excelId = $excel->id;
                    }
                }

                $history = new LeadsUpdateHistory();
                $history->lead_type = $request->leadType;
                $history->status = $request->status;
                $history->created_by = $request->user()->id;
                $history->just_lead_id = $justDialId;
                $history->crm_lead_id = $crmId;
                $history->sap_lead_id = $sapId;
                $history->excel_campaign_id = $excelId;
                if ($history->save()) {
                    DB::commit();
                    return $this->commonService->makeSuccessResponse('Status updated successfully', $data);
                }
            }

            return $this->commonService->makeErrorResponse('Failed to update status.');
        } catch (\Exception $e) {
            DB::rollback();
            return $this->commonService->makeErrorResponse($e->getMessage());
        }
    }

    function downloadPendingCalls()
    {
        $data = BeneficiaryImport::select('customer_name', 'mobile_number', 'invoice_date', 'description')
            ->where('status', 1)
            ->where('source', 1)
            ->get();
        $param = new stdClass;
        $param->type = 'pending-calls';
        return $this->excelExportService->exportExcel($param, $data);
    }
}
